"use client";

import React, { useState, useEffect } from "react";
import io from "socket.io-client";
import Sidebar from "@/app/Components/Home/Sidebar";
import Header from "@/app/Components/Home/Header";
import { getCookie } from "cookies-next";
import { decryptData } from "@/app/utils/DecryptData";

const ChatPageClient = ({ initialChats }) => {
  const [socket, setSocket] = useState(null);
  const [users, setUsers] = useState([]);
  const [selectedUser, setSelectedUser] = useState(null);
  const [messages, setMessages] = useState([]);
  const [newMessage, setNewMessage] = useState('');
  const [currentChat, setCurrentChat] = useState(null);
  const [newChatNotification, setNewChatNotification] = useState(''); 
  const token = getCookie("token");

  useEffect(() => {
    // Initialize Socket.IO connection with token
    const socketIo = io(`${process.env.NEXT_PUBLIC_API_URL}`, {
      auth: {
        token,
      },
    });
    setSocket(socketIo);

    socketIo.on('connect', () => {
      
    });

    // Listen for new chat notification
    socketIo.on('new chat', ({ client_id, public_key }) => {
      
      setNewChatNotification(`New chat initiated with client ${client_id}`);
      
      // Add the new client to users if not already present
      setUsers(prevUsers => {
        const isUserExists = prevUsers.some(user => user.id === client_id);
        if (!isUserExists) {
          return [...prevUsers, { id: client_id, name: `User ${client_id}`, public_key }];
        }
        return prevUsers;
      });
    });

    // Listen for incoming messages
    socketIo.on('chat message', (message) => {
      
      if (message.client_id === currentChat?.id) {
        setMessages(prevMessages => [...prevMessages, message]);
      }
    });

    return () => {
      socketIo.disconnect();
    };
  }, [currentChat]);

  const handleUserClick = (userId) => {
    const selectedChat = users.find(user => user.id === userId);
    if (socket && selectedChat) {
      // Emit the 'join chat' event with client_id and public_key
      socket.emit('join chat', {
        client_id: selectedChat.id,
        public_key: selectedChat.public_key,
      });

      // Fetch previous messages for the selected chat
      socket.emit('get messages', {
        client_id: selectedChat.id,
        public_key: selectedChat.public_key,
      });

      socket.on('messages', (encryptedData) => {
         const messages = decryptData(encryptedData)
        setMessages(messages);
      });

      setCurrentChat(selectedChat);
      setSelectedUser(userId);
      setNewChatNotification(''); // Clear any existing notifications
    }
  };

  const handleSendMessage = () => {
    if (socket && currentChat && newMessage.trim() !== '') {
      const message = {
        client_id: currentChat.id,
        public_key: currentChat.public_key,
        message: newMessage,
        sender: 'me'
      };
      

      socket.emit('chat message', message);
      setMessages(prevMessages => [...prevMessages, message]);
      setNewMessage('');
    }
  };

  return (
    <>
      <Sidebar />
      <div className="content">
        <Header />
        <div className="container-fluid top-bar">
          <div className="left-div">
            <h2>Chat</h2>
          </div>
          <div className="right-div"></div>
        </div>

        <div className="d-flex" style={{ height: "100vh" }}>
          <div
            className="border-end"
            style={{ width: "300px", padding: "10px", overflowY: "auto" }}
          >
            <h3>Chat Users</h3>
            <ul className="list-unstyled">
              {users.map((user) => (
                <li key={user.id} className="mb-2">
                  <button
                    onClick={() => handleUserClick(user.id)}
                    className={`btn w-100 text-start ${
                      selectedUser === user.id ? "btn-light" : "btn-outline-secondary"
                    }`}
                  >
                    {user.name}
                  </button>
                </li>
              ))}
            </ul>
          </div>

          <div className="flex-grow-1 p-3 d-flex flex-column">
            <h3>Chat</h3>
            {newChatNotification && (
              <div className="alert alert-info">
                {newChatNotification}
              </div>
            )}
            {selectedUser ? (
              <div className="flex-grow-1 d-flex flex-column">
                <h4>Chat with {users.find((user) => user.id === selectedUser)?.name}</h4>
                <div className="border p-3 mb-3 rounded" style={{ flex: 1, overflowY: "auto" }}>
                  {messages.length > 0 ? (
                    <ul className="list-unstyled">
                      {messages.map((msg, index) => (
                        <li
                          key={index}
                          className={`mb-2 d-flex ${
                            msg.sender === "me" ? "justify-content-end" : "justify-content-start"
                          }`}
                        >
                          <span
                            className={`p-2 rounded ${
                              msg.sender === "me" ? "bg-primary text-white" : "bg-light"
                            }`}
                          >
                            {msg.message}
                          </span>
                        </li>
                      ))}
                    </ul>
                  ) : (
                    <p>No messages found.</p>
                  )}
                </div>
                <div className="input-group mb-3">
                  <input
                    type="text"
                    className="form-control"
                    placeholder="Type a message"
                    value={newMessage}
                    onChange={(e) => setNewMessage(e.target.value)}
                    onKeyDown={(e) => e.key === "Enter" && handleSendMessage()}
                  />
                  <button className="btn btn-primary" onClick={handleSendMessage}>
                    Send
                  </button>
                </div>
              </div>
            ) : (
              <p>Select a user to start chatting.</p>
            )}
          </div>
        </div>
      </div>
    </>
  );
};

export default ChatPageClient;
